/**
  ******************************************************************************
  * @file    main_bl.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    14-July-2016
  * @brief   Entry point for the boot loader
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  ******************************************************************************  
  */ 


#include "stm32f4_discovery.h"

#include "bl.h"

SECTION_FLASH_MCB_BL const MCB g_BLMCB = 
{
    MCB_SIGNATURE_BL,
    {
        ((unsigned long)&_eidata - 0x8000000),
        (unsigned long)&_FLASH_PHYS_START,
        {BL_VERSION_MAJOR,BL_VERSION_MINOR,BL_VERSION_BETA,BL_VERSION_RESERVED},
    },
    {
        MIT_ENTRY_CT_BL,
        {
            (void *)DispatchIRQToBL,(void *)ToggleLEDViaBL,0,0
        }
    }
};

/** @brief The dispatcher from the appliation's IVT to the Bootloader's IVT. 
 *
 *  Every IRQ that the application does not handle explicitly must have an entry in the application's IVT
 *  that eventually ends up here. Typically that happens via an entry in the BL MIB.
 *
 *  @return none
 */


SECTION_FLASH_CODE_BL void DispatchIRQToBL(void)
{
    unsigned long a_VecNo = *((unsigned long *)0xe000ed04) & 0x1FF; // SCB->ICSR &SCB_ICSR_VECTACTIVE_Msk  
    (*g_pfnVectorsBL[a_VecNo])();
}

/** @brief Example for a service function in the BL that the application can call via the MIB. Toggles an LED.
 *
 *  @param p_LED LED number
 *  @return none
 */

void ToggleLEDViaBL(unsigned long p_LED)
{
    STM_EVAL_LEDToggle((Led_TypeDef)p_LED);
}

/** @brief Skeleton for a routine to verify the correctness of an application image.
 *
 *  @param p_MCB Pointer to the exptected MCB of a loaded application.
 *  @return none
 */

unsigned long VerifyAppImage(const MCB *p_MCB)
{
    if (p_MCB->m_Sig != MCB_SIGNATURE_FW)
        return 0;
    // TODO: Verify checksum over image
    return 1;
}

/** @brief Bootloader infite loop.
 *
 *  Called when no valid application image is loaded. In real life, this must accept Firmware Download
 *  packets over a communication interface and program the flash with the received data.
 *  @return none
 */

void BL_MainLoadLoop(void)
{
    while(1) 
    {
        volatile unsigned long a_BusyWaitCt = 25000000;
        STM_EVAL_LEDToggle(LED4);
        while (--a_BusyWaitCt);
    };
}

/** @brief Boot loader entry point, called from the reset vector after minimal system setup 
 *
 *  @param p_MCB Pointer to the exptected MCB of a loaded application.
 *  @return none
 */

int main_BL(void)
{
    if (VerifyAppImage(&g_fw_mcb))
    {
        // reprogram the IVT. The Instruction Barrier is necessary here (cf. ARM documentation)
        __asm volatile
        (
            " cpsid i    \n"
            " isb        \n"
        );
        *((unsigned long *)0xe000ed08) = ((unsigned long)(&g_pfnVectorsFWStubs) & (unsigned long)0x1FFFFF80);   // set SCB->VTOR

        // redirect stack pointer to vector 0 in the application's IVT
        __asm volatile 
        (
            " ldr r0,pcrelSP  \n"
            " ldr r13,[r0]    \n"
            " .align 2        \n"
            "pcrelSP: .word g_pfnVectorsFWStubs \n"
         );
         // dispatch to app entry
        (g_pfnVectorsFWStubs[1])();
    }
    else
        BL_MainLoadLoop();
}

